<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\JsonResponse;
use Hash;
use App\Helpers\Helper;

class LoginController extends Controller
{
    /**
     * Register api
     *
     * @return \Illuminate\Http\Response
     */
    public function register(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'name' => 'required',
            // 'father_name' => 'required',
            'mobile_no' => 'required|string|numeric|digits:10|unique:users',
            'address' => 'required',
            // 'city' => 'required',
            // 'state' => 'required',
            // 'pincode' => 'required',
            // 'pan' => 'required',
            'password' => 'required|string|min:8'
        ]);

        if ($validate->fails()) {
            return response()->json([
                'status' => false,
                'message' => 'Validation Error!',
                'data' => $validate->errors(),
            ], 400);
        }
        
        $otp  = rand(111111,999999);

        $user = new User;
        $user->name = $request->input('name');
        $user->father_name = $request->input('father_name');
        $user->mobile_no = $request->input('mobile_no');
        $user->address = $request->input('address');
        $user->city = $request->input('city');
        $user->state = $request->input('state');
        $user->pincode = $request->input('pincode');
        $user->pan = $request->input('pan');
        $user->password = Hash::make($request->input('password'));
        $user->firebase_token = $request->input('firebase_token');
        $user->save();

        $token = $user->createToken('api_token')->plainTextToken;
        $user->remember_token = $token;
        $user->otp = $otp;
        $user->save();

        $data['token'] = $token;
        $data['user'] = $user;
        
        $sms = Helper::sendSms($otp, $request->input('mobile_no'));
        

        $response = [
            'status' => true,
            'message' => 'User is created successfully.',
            'data' => $data,
        ];

        return response()->json($response, 200);
    }


    /**
     * Login api
     *
     * @return \Illuminate\Http\Response
     */
    public function login(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'mobile_no' => 'required|string|numeric|digits:10',
            'password' => 'required|string|min:8'
        ]);

        if($validate->fails()){
            return response()->json([
                'status' => false,
                'message' => 'Validation Error!',
                'data' => $validate->errors(),
            ], 400);
        }

        // Check mobile_no exist
        $user = User::where('mobile_no', $request->input('mobile_no'))->first();

        // Check password
        if(!$user || !Hash::check($request->input('password'), $user->password)) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid credentials'
                ], 400);
        }

        $data['token'] = $user->createToken($request->mobile_no)->plainTextToken;
        $user->remember_token = $data['token'];
        $user->firebase_token = $request->input('firebase_token');
        $user->save();
        $data['user'] = $user;

        $response = [
            'status' => true,
            'message' => 'User is logged in successfully.',
            'data' => $data,
        ];

        return response()->json($response, 200);
    }
    
    
    /**
     * forgot api
     *
     * @return \Illuminate\Http\Response
     */
    public function forgot(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'mobile_no' => 'required|string|numeric|digits:10',
            'otp' => 'required|string|numeric|digits:6',
        ]);

        if($validate->fails()){
            return response()->json([
                'status' => false,
                'message' => 'Validation Error!',
                'data' => $validate->errors(),
            ], 400);
        }

        // Check mobile_no exist
        $user = User::where('mobile_no', $request->input('mobile_no'))->where('otp', $request->input('otp'))->first();

        // Check 
        if(!$user) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid OTP'
                ], 400);
        }
        
        $user->mobile_verified = 1;
        $user->status = 1;
        $user->save();

        $response = [
            'status' => true,
            'data' => $user,
        ];

        return response()->json($response, 200);
    }
    
    
    /**
     * change password api
     *
     * @return \Illuminate\Http\Response
     */
     
    public function changePassword(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'new_password' => 'required|string|min:8',
            'new_password_confirmation' => 'required|string|min:8|same:new_password',
            'mobile_no' => 'required|exists:users,mobile_no',
        ]);
        
        if($validate->fails()){
            return response()->json([
                'status' => false,
                'message' => 'Validation Error!',
                'data' => $validate->errors(),
            ], 400);
        }

        $user = User::where('mobile_no', $request->mobile_no)->first();

        if ($user) {
            $user->update([
                'password' => Hash::make($request->new_password),
            ]);

            $response = [
                'status' => true,
                'message' => 'Password changed successfully.',
            ];
    
            return response()->json($response, 200);
        }

    }
     
     
     
     
     
     
     
     

    /**
     * Log out the user from application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        auth()->user()->tokens()->delete();
        return response()->json([
            'status' => true,
            'message' => 'User is logged out successfully'
            ], 200);
    }
}
